// $Id: CSystemInformation.cpp,v 1.2 2007/02/08 21:06:44 paul Exp $

/*
 * All contents of this source code are copyright 2005 Exp Digital Uk.
 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy
 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
 * All content is the Intellectual property of Exp Digital Uk.
 * Certain sections of this code may come from other sources. They are credited where applicable.
 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
 */

#include "CSystemInformation.hpp"
using Exponent::Host::CSystemInformation;

#ifdef WIN32
	#pragma comment(lib, "Winmm.lib")
#else
	#include <mach-o/arch.h>
#endif

//	===========================================================================
void CSystemInformation::getSystemInfo(SSystemInformation &sys)
{
	sys.m_computerName = getComputerName();
	sys.m_userName     = getUserName();
	getOperatingSystemVersion(sys.m_osVersion);
	sys.m_processorString = getCpuNameString();
	getOsVersionInfo(sys.m_major, sys.m_minor, sys.m_build);
	sys.m_cpuSpeed			 = getCpuSpeed();
	sys.m_numberOfProcessors = getNumberOfProcessors();
}

//	===========================================================================
void CSystemInformation::dumpSystemInfoToFile(CSystemString &path)
{
	// Check that the path is correct..
	if (!path.hasExtension())
	{
		path.appendString("CPUStats.txt");
	}

	// Create the stream
	CTextStream stream(path, CTextStream::e_output);

	// Output
	dumpSystemInfoToFile(stream);

	// Close the file
	stream.closeStream();
}

//	===========================================================================
void CSystemInformation::dumpSystemInfoToFile(CTextStream &stream)
{
	if (stream.isStreamOpen())
	{
		// Get the system info
		SSystemInformation sys;
		getSystemInfo(sys);

		stream << "Exponent Host info extraction tool\n";
		stream << "----------------------------------\n";
		stream << "System name\t\t:\t\t" << sys.m_computerName << "\n";
		stream << "User name\t\t:\t\t" << sys.m_userName << "\n";
		stream << "Operating system\t:\t\t" << sys.m_osVersion << "\n";
		stream << "Processor\t\t:\t\t" << sys.m_processorString << "\n";
		stream << "Cpu Speed\t\t:\t\t" << sys.m_cpuSpeed << "\n";
		stream << "OS version\t\t:\t\t" << sys.m_major << "." << sys.m_minor << "." << sys.m_build << "\n";
		stream << "Number of processors\t:\t\t" << sys.m_numberOfProcessors << "\n";
		stream << "----------------------------------\n";
	}
}

//	===========================================================================
long CSystemInformation::getNumberOfProcessors()
{
#ifdef WIN32
	SYSTEM_INFO sysInfo;
	GetSystemInfo(&sysInfo);
	return (long)sysInfo.dwNumberOfProcessors;
#else
	return (long)MPProcessors();
#endif
}

//	===========================================================================
CString CSystemInformation::getComputerName()
{
#ifdef WIN32
	unsigned long length  = 120;
	char *buffer = new char[length];
	GetComputerName(buffer, &length);
	buffer[length-1] = '\0';
	CString string = buffer;
	FREE_ARRAY_POINTER(buffer);
	return string;
#else
	CFStringRef computerName = CSCopyMachineName();
	CString string;
	string.setStringFromCFStringRef(computerName);
	CFRelease(computerName);
	return string;
#endif
}

//	===========================================================================
CString CSystemInformation::getUserName()
{
#ifdef WIN32
	unsigned long length  = 120;
	char *buffer = new char[length];
	GetUserName(buffer, &length);
	buffer[length-1] = '\0';
	CString string = buffer;
	FREE_ARRAY_POINTER(buffer);
	return string;
#else
	CFStringRef userName = CSCopyUserName(FALSE);
	CString string;
	string.setStringFromCFStringRef(userName);
	CFRelease(userName);
	return string;
#endif
}

//	===========================================================================
bool CSystemInformation::getOperatingSystemVersion(CString &string)
{
#ifdef WIN32
	OSVERSIONINFOEX osvi;
	BOOL bOsVersionInfoEx;

	// Try calling GetVersionEx using the OSVERSIONINFOEX structure.
	// If that fails, try using the OSVERSIONINFO structure.

	memset(&osvi, 0, sizeof(OSVERSIONINFOEX));
	osvi.dwOSVersionInfoSize = sizeof(OSVERSIONINFOEX);

	if( !(bOsVersionInfoEx = GetVersionEx ((OSVERSIONINFO *) &osvi)) )
	{
		osvi.dwOSVersionInfoSize = sizeof (OSVERSIONINFO);
		if (! GetVersionEx ( (OSVERSIONINFO *) &osvi) )
		{
			return false;
		}
	}

	switch (osvi.dwPlatformId)
	{
		// Test for the Windows NT product family.
		case VER_PLATFORM_WIN32_NT:

			// Test for the specific product family.
			if ( osvi.dwMajorVersion == 5 && osvi.dwMinorVersion == 2 )
			{
         		string = "Windows Server 2003";
			}

			if ( osvi.dwMajorVersion == 5 && osvi.dwMinorVersion == 1 )
			{
            		string = "Windows XP";
			}

			if ( osvi.dwMajorVersion == 5 && osvi.dwMinorVersion == 0 )
			{
            		string = "Windows 2000";
			}

			if ( osvi.dwMajorVersion <= 4 )
			{
            		string = "Windows NT";
			}

			// Test for specific product on Windows NT 4.0 SP6 and later.
			if( bOsVersionInfoEx )
			{
				// Test for the workstation type.
				if ( osvi.wProductType == VER_NT_WORKSTATION )
				{
					if( osvi.dwMajorVersion == 4 )
					{
						string += " Workstation 4.0";
					}
					else if( osvi.wSuiteMask & VER_SUITE_PERSONAL )
					{
						string += " Home Edition";
					}
					else
					{
						string += " Professional";
					}
				}

				// Test for the server type.
				else if ( osvi.wProductType == VER_NT_SERVER )
				{
				if( osvi.dwMajorVersion == 5 && osvi.dwMinorVersion == 2 )
				{
					if( osvi.wSuiteMask & VER_SUITE_DATACENTER )
					{
						string += " Datacenter Edition";
					}
					else if( osvi.wSuiteMask & VER_SUITE_ENTERPRISE )
					{
						string += " Enterprise Edition";
					}
					else if ( osvi.wSuiteMask == VER_SUITE_BLADE )
					{
						string += " Web Edition";
					}
					else
					{
						string += " Standard Edition";
					}
				}

				else if( osvi.dwMajorVersion == 5 && osvi.dwMinorVersion == 0 )
				{
					if( osvi.wSuiteMask & VER_SUITE_DATACENTER )
					{
						string += " Datacenter Server";
					}
					else if( osvi.wSuiteMask & VER_SUITE_ENTERPRISE )
					{
						string += " Advanced Server";
					}
					else
					{
						string += " Server";
					}
				}

				else  // Windows NT 4.0
				{
					if( osvi.wSuiteMask & VER_SUITE_ENTERPRISE )
					{
						string += " Server 4.0, Enterprise Edition";
					}
					else
					{
						string += " Server 4.0";
					}
				}
				}
			}
			else  // Test for specific product on Windows NT 4.0 SP5 and earlier
			{
				string = "Pre Windows NT 4.0 SP5";
			}
			break;

		// Test for the Windows 95 product family.
		case VER_PLATFORM_WIN32_WINDOWS:
			string = "Windows 95 / 98 / ME";
		break;

		case VER_PLATFORM_WIN32s:
			string = "Win32";
		break;
	}
	return true;
#else
	string = "Mac OSX";
	return true;
#endif
}

//	===========================================================================
void CSystemInformation::getOsVersionInfo(long &major, long &minor, long &build)
{
#ifdef WIN32
	OSVERSIONINFO osInfo;
	osInfo.dwOSVersionInfoSize = sizeof(OSVERSIONINFO);
	GetVersionEx(&osInfo);
	major = osInfo.dwMajorVersion;
	minor = osInfo.dwMinorVersion;
	build = osInfo.dwBuildNumber;
#else
	long version = 0;
	Gestalt(gestaltSystemVersion, &version);

	major = (version & 0xFF00) >> 8;
	minor = (version & 0xFF00) >> 4;
	build = (version & 0x000F);
#endif
}

//	===========================================================================
CString CSystemInformation::getCpuNameString()
{
#ifdef WIN32
	char szName[64] = {0};

	__asm
	{
		mov      eax, 0x80000002
		CPUID
		mov      DWORD PTR [szName+0],eax
		mov      DWORD PTR [szName+4],ebx
		mov      DWORD PTR [szName+8],ecx
		mov      DWORD PTR [szName+12],edx
		mov      eax, 0x80000003
		CPUID
		mov      DWORD PTR [szName+16],eax
		mov      DWORD PTR [szName+20],ebx
		mov      DWORD PTR [szName+24],ecx
		mov      DWORD PTR [szName+28],edx
		mov      eax, 0x80000004
		CPUID
		mov      DWORD PTR [szName+32],eax
		mov      DWORD PTR [szName+36],ebx
		mov      DWORD PTR [szName+40],ecx
		mov      DWORD PTR [szName+44],edx
	}

	CString string = szName;
	string.removeTrailingAndLeadingWhiteSpace();
	return string;
#else
	const NXArchInfo *info = NXGetLocalArchInfo();
	CString string = info->name;
	return string;
#endif
}

//	===========================================================================
long CSystemInformation::getCpuSpeed()
{
#ifdef WIN32
	int   nTimeStart = 0;
	int   nTimeStop = 0;
	DWORD dwStartTicks = 0;
	DWORD dwEndTicks = 0;
	DWORD dwTotalTicks = 0;
	DWORD dwCpuSpeed = 0;

	nTimeStart = timeGetTime();

	for(;;)
	{
		nTimeStop = timeGetTime();

		if ((nTimeStop - nTimeStart) > 1)
		{
			dwStartTicks = determineTimeStamp();
			break;
		}
	}

	nTimeStart = nTimeStop;

	for(;;)
	{
		nTimeStop = timeGetTime();
		if ((nTimeStop - nTimeStart) > 500)    // one-half second
		{
			dwEndTicks = determineTimeStamp();
			break;
		}
	}

	dwTotalTicks = dwEndTicks - dwStartTicks;
	dwCpuSpeed = dwTotalTicks / 500000;

	return (long)(dwCpuSpeed);
#else
	return GetCPUSpeed();
#endif
}

#ifdef WIN32
//	===========================================================================
DWORD CSystemInformation::determineTimeStamp()
{
	DWORD dwTickVal;

	__asm
	{
		_emit 0Fh
		_emit 31h
		mov   dwTickVal, eax
	}

	return (dwTickVal);
}
#endif